<?php

/**
 * Created by PhpStorm.
 * User: tnagy
 * Date: 2019.08.06.
 * Time: 11:09
 */
class MM_WPFS_CheckoutSubmissionService {

	const POPUP_FORM_SUBMIT_STATUS_CREATED = 'created';
	const POPUP_FORM_SUBMIT_STATUS_PENDING = 'pending';
	const POPUP_FORM_SUBMIT_STATUS_FAILED = 'failed';
	const POPUP_FORM_SUBMIT_STATUS_CANCELLED = 'cancelled';
	const POPUP_FORM_SUBMIT_STATUS_SUCCESS = 'success';
	const POPUP_FORM_SUBMIT_STATUS_COMPLETE = 'complete';
	const CHECKOUT_SESSION_STATUS_SUCCESS = 'success';
	const CHECKOUT_SESSION_STATUS_CANCELLED = 'cancelled';
	const PROCESS_RESULT_SET_TO_SUCCESS = 1;
	const PROCESS_RESULT_SET_TO_FAILED = 2;
	const PROCESS_RESULT_EXPIRED = 3;
	const PROCESS_RESULT_WAIT_FOR_STATUS_CHANGE = 4;
	const PROCESS_RESULT_INTERNAL_ERROR = 20;
	const ACTION_FULLSTRIPE_PROCESS_CHECKOUT_SUBMISSIONS = 'fullstripe_process_checkout_submissions';
	const STRIPE_CALLBACK_PARAM_WPFS_POPUP_FORM_SUBMIT_HASH = 'wpfs-sid';
	const STRIPE_CALLBACK_PARAM_WPFS_CHECKOUT_SESSION_ID = 'wpfs-csid';
	const STRIPE_CALLBACK_PARAM_WPFS_STATUS = 'wpfs-status';

	/** @var bool */
	private static $running = false;
	/** @var bool */
	private $debugLog = false;
	/** @var $db MM_WPFS_Database */
	private $db = null;
	/** @var int Iteration count for processing entries in the scheduled function */
	private $iterationCount = 5;
	/** @var int Entry count processed in one iteration */
	private $entryCount = 50;

	/**
	 * MM_WPFS_CheckoutSubmissionService constructor.
	 */
	public function __construct() {
		if ( $this->debugLog ) {
			MM_WPFS_Utils::log( 'MM_WPFS_CheckoutSubmissionService->__construct(): CALLED, running=' . ( $this->isRunning() ? 'true' : 'false' ) );
		}
		$this->setup();
		$this->hooks();
	}

	/**
	 * @return bool
	 */
	private function isRunning() {
		return self::$running;
	}

	/**
	 *
	 */
	private function setup() {
		$this->db = new MM_WPFS_Database();
	}

	private function hooks() {
		add_action(
			self::ACTION_FULLSTRIPE_PROCESS_CHECKOUT_SUBMISSIONS,
			array(
				$this,
				'processCheckoutSubmissions'
			)
		);
	}

	public static function onActivation() {
		if ( ! wp_next_scheduled( self::ACTION_FULLSTRIPE_PROCESS_CHECKOUT_SUBMISSIONS ) ) {
			wp_schedule_event( time(), WP_FULL_STRIPE_CRON_SCHEDULES_KEY_15_MIN, self::ACTION_FULLSTRIPE_PROCESS_CHECKOUT_SUBMISSIONS );
			MM_WPFS_Utils::log( 'MM_WPFS_CheckoutSubmissionService->onActivation(): Event scheduled.' );
		}
	}

	public static function onDeactivation() {
		wp_clear_scheduled_hook( self::ACTION_FULLSTRIPE_PROCESS_CHECKOUT_SUBMISSIONS );
		MM_WPFS_Utils::log( 'MM_WPFS_CheckoutSubmissionService->onDeactivation(): Scheduled event cleared.' );
	}

	/**
	 * @param MM_WPFS_Public_PopupSubscriptionFormModel $subscriptionFormModel
	 *
	 * @return \Stripe\Checkout\Session
	 */
	public function createCheckoutSessionBySubscriptionForm( $subscriptionFormModel ) {
		if ( $this->debugLog ) {
			MM_WPFS_Utils::log( 'createCheckoutSessionBySubscriptionForm(): CALLED, params: subscriptionFormModel=' . print_r( $subscriptionFormModel, true ) );
		}

		$submitHash = $this->createSubmitEntry( $subscriptionFormModel );

		$checkoutSessionParameters = $this->prepareCheckoutSessionParameterDataForSubscription( $submitHash, $subscriptionFormModel );

		$stripeCheckoutSession = \Stripe\Checkout\Session::create( $checkoutSessionParameters );

		$this->updateSubmitEntryWithSessionIdToPending( $submitHash, $stripeCheckoutSession->id );

		return $stripeCheckoutSession;

	}

	/**
	 * @param MM_WPFS_Public_FormModel $formModel
	 *
	 * @return string
	 */
	private function createSubmitEntry( $formModel ) {
		$options    = get_option( 'fullstripe_options' );
		$liveMode   = $options['apiMode'] === 'live';
		$salt       = wp_generate_password( 16, false );
		$submitId   = time() . '|' . $formModel->getFormHash() . '|' . $liveMode . '|' . $salt;
		$submitHash = hash( 'sha256', $submitId );
		$referrer   = esc_url(
			add_query_arg(
				array(
					self::STRIPE_CALLBACK_PARAM_WPFS_POPUP_FORM_SUBMIT_HASH => $submitHash
				),
				home_url( $formModel->getReferrer() )
			) . '#' . \MM_WPFS_FormViewConstants::ATTR_ID_VALUE_PREFIX . $formModel->getFormHash()
		);
		$this->db->insert_popup_form_submit(
			$submitHash,
			$formModel->getFormHash(),
			MM_WPFS_Utils::getFormType( $formModel->getForm() ),
			$referrer,
			json_encode( $formModel->getPostData(), JSON_UNESCAPED_UNICODE ),
			$liveMode
		);

		return $submitHash;
	}

	/**
	 * @param string $submitHash
	 * @param MM_WPFS_Public_PopupSubscriptionFormModel $subscriptionFormModel
	 *
	 * @return array
	 */
	private function prepareCheckoutSessionParameterDataForSubscription( $submitHash, $subscriptionFormModel ) {
		$collectBillingAddress = isset( $subscriptionFormModel->getForm()->showBillingAddress ) && 1 == $subscriptionFormModel->getForm()->showBillingAddress ? 'required' : 'auto';
		$subscriptionDataArray = array(
			'items' => array(
				array(
					'plan'     => $subscriptionFormModel->getStripePlan()->id,
					'quantity' => 1
				)
			)
		);
		$trialPeriodDays       = MM_WPFS_Utils::get_trial_period_days_for_plan( $subscriptionFormModel->getStripePlan() );
		if ( ! is_null( $trialPeriodDays ) ) {
			$subscriptionDataArray['trial_period_days'] = $trialPeriodDays;
		}
		$checkoutSessionParameters = array(
			'payment_method_types'       => array( 'card' ),
			'mode'                       => 'subscription',
			'subscription_data'          => $subscriptionDataArray,
			'client_reference_id'        => $submitHash,
			'billing_address_collection' => $collectBillingAddress,
			'success_url'                => $this->buildCheckoutSessionSuccessURL( $submitHash ),
			'cancel_url'                 => $this->buildCheckoutSessionCancelURL( $submitHash )
		);
		$options                   = get_option( 'fullstripe_options' );
		$lock_email                = $options['lock_email_field_for_logged_in_users'];
		$email_address             = '';
		$is_user_logged_in         = is_user_logged_in();
		if ( '1' == $lock_email && $is_user_logged_in ) {
			$current_user  = wp_get_current_user();
			$email_address = $current_user->user_email;
		}
		if ( ! empty( $email_address ) ) {
			$checkoutSessionParameters['customer_email'] = $email_address;
		}
		if ( isset( $subscriptionFormModel->getForm()->preferredLanguage ) ) {
			$checkoutSessionParameters['locale'] = $subscriptionFormModel->getForm()->preferredLanguage;
		}
		$setupFee = MM_WPFS_Utils::get_setup_fee_for_plan( $subscriptionFormModel->getStripePlan() );
		if ( $setupFee > 0 ) {
			$checkoutSessionParameters['line_items'] = array(
				array(
					'amount'      => $setupFee,
					'currency'    => $subscriptionFormModel->getStripePlan()->currency,
					'name'        => __( 'One-time setup fee', 'wp-full-stripe' ),
					'description' => sprintf(
						__( 'Subscription plan: %s', 'wp-full-stripe' ),
						$subscriptionFormModel->getStripePlan()->id
					),
					'quantity'    => 1
				)
			);
		}

		return $checkoutSessionParameters;
	}

	/**
	 * @param $submitHash
	 *
	 * @return string
	 */
	private function buildCheckoutSessionSuccessURL( $submitHash ) {
		return $this->buildCheckoutSessionStatusURL( $submitHash, self::CHECKOUT_SESSION_STATUS_SUCCESS );
	}

	/**
	 * @param $submitHash
	 * @param $status
	 *
	 * @return string
	 */
	private function buildCheckoutSessionStatusURL( $submitHash, $status ) {
		return add_query_arg(
			array(
				'action'                                                => 'wp_full_stripe_handle_checkout_session',
				self::STRIPE_CALLBACK_PARAM_WPFS_POPUP_FORM_SUBMIT_HASH => $submitHash,
				// self::STRIPE_CALLBACK_PARAM_WPFS_CHECKOUT_SESSION_ID    => '{CHECKOUT_SESSION_ID}',
				self::STRIPE_CALLBACK_PARAM_WPFS_STATUS                 => $status
			),
			admin_url( 'admin-ajax.php' )
		);
	}

	/**
	 * @param $submitHash
	 *
	 * @return string
	 */
	protected function buildCheckoutSessionCancelURL( $submitHash ) {
		return $this->buildCheckoutSessionStatusURL( $submitHash, self::CHECKOUT_SESSION_STATUS_CANCELLED );
	}

	/**
	 * @param $submitHash
	 * @param $stripeCheckoutSessionId
	 *
	 * @return false|int
	 */
	public function updateSubmitEntryWithSessionIdToPending( $submitHash, $stripeCheckoutSessionId ) {
		return $this->db->update_popup_form_submit_by_hash(
			$submitHash,
			array(
				'checkoutSessionId' => $stripeCheckoutSessionId,
				'status'            => self::POPUP_FORM_SUBMIT_STATUS_PENDING
			)
		);
	}

	/**
	 * @param MM_WPFS_Public_PopupPaymentFormModel $paymentFormModel
	 *
	 * @return \Stripe\Checkout\Session
	 */
	public function createCheckoutSessionByPaymentForm( $paymentFormModel ) {
		if ( $this->debugLog ) {
			MM_WPFS_Utils::log( 'createCheckoutSessionByPaymentForm(): CALLED, params: paymentFormModel=' . print_r( $paymentFormModel, true ) );
		}

		$submitHash = $this->createSubmitEntry( $paymentFormModel );

		$checkoutSessionParameters = $this->prepareCheckoutSessionParameterDataForPayment( $submitHash, $paymentFormModel );

		$stripeCheckoutSession = \Stripe\Checkout\Session::create( $checkoutSessionParameters );

		$this->updateSubmitEntryWithSessionIdToPending( $submitHash, $stripeCheckoutSession->id );

		return $stripeCheckoutSession;
	}

	/**
	 * @param string $submitHash
	 * @param MM_WPFS_Public_PopupPaymentFormModel $paymentFormModel
	 *
	 * @return array
	 */
	private function prepareCheckoutSessionParameterDataForPayment( $submitHash, $paymentFormModel ) {
		$collectBillingAddress     = isset( $paymentFormModel->getForm()->showBillingAddress ) && 1 == $paymentFormModel->getForm()->showBillingAddress ? 'required' : 'auto';
		$checkoutSessionParameters = array(
			'payment_method_types'       => array( 'card' ),
			'client_reference_id'        => $submitHash,
			'billing_address_collection' => $collectBillingAddress,
			'success_url'                => $this->buildCheckoutSessionSuccessURL( $submitHash ),
			'cancel_url'                 => $this->buildCheckoutSessionCancelURL( $submitHash )
		);
		if ( MM_WPFS::PAYMENT_TYPE_CARD_CAPTURE === $paymentFormModel->getForm()->customAmount ) {
			$checkoutSessionParameters['mode'] = 'setup';
		} else {
			$checkoutSessionParameters['mode'] = 'payment';
			$lineItem                          = array(
				'amount'   => $paymentFormModel->getAmount(),
				'currency' => $paymentFormModel->getForm()->currency,
				'quantity' => 1,
				'name'     => $paymentFormModel->getProductName()
			);
			if ( isset( $paymentFormModel->getForm()->companyName ) && ! empty( $paymentFormModel->getForm()->companyName ) ) {
				$lineItem['description'] = $paymentFormModel->getForm()->companyName;
			}
			$imagesValueArray = $this->prepareImagesValueArray( $paymentFormModel->getForm() );
			if ( count( $imagesValueArray ) > 0 ) {
				$lineItem['images'] = $imagesValueArray;
			}
			$checkoutSessionParameters['line_items'] = array( $lineItem );
			$captureAmount                           = MM_WPFS_Utils::prepareCaptureIntentByFormModel( $paymentFormModel );
			if ( false === $captureAmount ) {
				$checkoutSessionParameters['payment_intent_data'] = array(
					'capture_method' => 'manual'
				);
			}
		}
		// tnagy get logged in user's email address
		$options        = get_option( 'fullstripe_options' );
		$lockEmail      = $options['lock_email_field_for_logged_in_users'];
		$emailAddress   = '';
		$isUserLoggedIn = is_user_logged_in();
		if ( '1' == $lockEmail && $isUserLoggedIn ) {
			$currentUser  = wp_get_current_user();
			$emailAddress = $currentUser->user_email;
		}
		if ( ! empty( $emailAddress ) ) {
			$checkoutSessionParameters['customer_email'] = $emailAddress;
		}
		if ( isset( $paymentFormModel->getForm()->preferredLanguage ) ) {
			$checkoutSessionParameters['locale'] = $paymentFormModel->getForm()->preferredLanguage;
		}

		return $checkoutSessionParameters;
	}

	/**
	 * @param $popupPaymentForm
	 *
	 * @return array
	 */
	private function prepareImagesValueArray( $popupPaymentForm ) {
		$imagesValue = array();
		if ( empty( $popupPaymentForm->image ) ) {
			// $imagesValue = [ self::DEFAULT_CHECKOUT_LINE_ITEM_IMAGE ];
		} else {
			array_push( $imagesValue, $popupPaymentForm->image );
		}

		return $imagesValue;
	}

	/**
	 * @param $submitHash
	 *
	 * @return array|null|object|void
	 */
	public function retrieveSubmitEntry( $submitHash ) {
		return $this->db->find_popup_form_submit_by_hash( $submitHash );
	}

	public function processCheckoutSubmissions() {
        if ( $this->debugLog ) {
            MM_WPFS_Utils::log('MM_WPFS_CheckoutSubmissionService->processCheckoutSubmissions(): CALLED');
        }
		try {

			if ( ! $this->isRunning() ) {
				$this->start();
				$this->findAndProcessSubmissions();
				$this->stop();
			}

		} catch ( Exception $e ) {
			MM_WPFS_Utils::logException( $e, $this );
			$this->stop();
		}
        if ( $this->debugLog ) {
            MM_WPFS_Utils::log('MM_WPFS_CheckoutSubmissionService->checkCheckoutSubmissions(): FINISHED');
        }
	}

	private function start() {
		self::$running = true;
	}

	private function findAndProcessSubmissions() {
		$iteration                  = 0;
		$popupFormSubmitsToProcess  = array();
		$popupFormSubmitsToComplete = array();
		$popupFormSubmitIdsToDelete = array();
		$popupFormSubmitsTouched    = array();

		$popupFormSubmits = $this->findPopupEntries();
		if ( isset( $popupFormSubmits ) ) {

			if ( $this->debugLog ) {
				MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Found ' . count( $popupFormSubmits ) . ' record(s) to process.' );
			}

			while ( $iteration < $this->iterationCount && count( $popupFormSubmits ) > 0 ) {
				$iteration ++;

				// tnagy prepare array of submits
				if ( ! is_array( $popupFormSubmits ) ) {
					$popupFormSubmits = array( $popupFormSubmits );
				}

				// tnagy sort out submits by status
				foreach ( $popupFormSubmits as $popupFormSubmit ) {

					if ( ! array_key_exists( $popupFormSubmit->id, $popupFormSubmitsTouched ) ) {

						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Processing Checkout Form Submission=' . print_r( $popupFormSubmit, true ) );
						}

						// tnagy mark record as touched
						$popupFormSubmitsTouched[ $popupFormSubmit->id ] = $popupFormSubmit;

						if ( self::POPUP_FORM_SUBMIT_STATUS_CREATED === $popupFormSubmit->status ) {
							array_push( $popupFormSubmitsToProcess, $popupFormSubmit );
							if ( $this->debugLog ) {
								MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to PROCESS.' );
							}
						} elseif ( self::POPUP_FORM_SUBMIT_STATUS_PENDING === $popupFormSubmit->status ) {
							array_push( $popupFormSubmitsToProcess, $popupFormSubmit );
							if ( $this->debugLog ) {
								MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to PROCESS.' );
							}
						} elseif ( self::POPUP_FORM_SUBMIT_STATUS_SUCCESS === $popupFormSubmit->status ) {
							array_push( $popupFormSubmitsToComplete, $popupFormSubmit );
							if ( $this->debugLog ) {
								MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to PROCESS.' );
							}
						} elseif ( self::POPUP_FORM_SUBMIT_STATUS_FAILED === $popupFormSubmit->status ) {
							array_push( $popupFormSubmitIdsToDelete, $popupFormSubmit->id );
							if ( $this->debugLog ) {
								MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to DELETE.' );
							}
						} elseif ( self::POPUP_FORM_SUBMIT_STATUS_COMPLETE === $popupFormSubmit->status ) {
							array_push( $popupFormSubmitIdsToDelete, $popupFormSubmit->id );
							if ( $this->debugLog ) {
								MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to DELETE.' );
							}
						}
					}

				}

				// tnagy process submits
				foreach ( $popupFormSubmitsToProcess as $popupFormSubmit ) {
					$result = $this->processSinglePopupFormSubmit( $popupFormSubmit );
					if ( self::PROCESS_RESULT_SET_TO_SUCCESS === $result ) {
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Checkout Form Submission successfully processed.' );
						}
						array_push( $popupFormSubmitsToComplete, $popupFormSubmit );
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to COMPLETE.' );
						}
					} elseif ( self::PROCESS_RESULT_SET_TO_FAILED === $result ) {
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Checkout Form Submission processing failed.' );
						}
						array_push( $popupFormSubmitIdsToDelete, $popupFormSubmit->id );
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to DELETE.' );
						}
					} elseif ( self::PROCESS_RESULT_EXPIRED === $result ) {
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): CheckoutSession expired for Checkout Form Submission.' );
						}
						$this->updateSubmitEntryWithCancelled( $popupFormSubmit );
						array_push( $popupFormSubmitIdsToDelete, $popupFormSubmit->id );
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to DELETE.' );
						}
					} elseif ( self::PROCESS_RESULT_WAIT_FOR_STATUS_CHANGE === $result ) {
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Checkout Form Submission skipped, waiting for status change.' );
						}
					} elseif ( self::PROCESS_RESULT_INTERNAL_ERROR === $result ) {
						MM_WPFS_Utils::log( 'findAndProcessSubmissions(): ERROR: Internal error occurred during Checkout Form Submission.' );
					}
				}

				// tnagy complete submits
				foreach ( $popupFormSubmitsToComplete as $popupFormSubmit ) {
					$this->updateSubmitEntryWithComplete( $popupFormSubmit );
					array_push( $popupFormSubmitIdsToDelete, $popupFormSubmit->id );
					if ( $this->debugLog ) {
						MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Proposed to DELETE.' );
					}
				}

				// tnagy delete submits
				$deleted = $this->deleteSubmitEntriesById( $popupFormSubmitIdsToDelete );

				// tnagy clear arrays
				$popupFormSubmitsToProcess  = array();
				$popupFormSubmitsToComplete = array();
				$popupFormSubmitIdsToDelete = array();

				if ( $this->debugLog ) {
					MM_WPFS_Utils::log( 'findAndProcessSubmissions(): Deleted ' . $deleted . ' Checkout Form Submission(s)' );
				}

				// tnagy load next fragment of submits
				$popupFormSubmits = $this->findPopupEntries();
			}
		}

	}

	/**
	 * @return array|null|object
	 */
	private function findPopupEntries() {
		return $this->db->find_popup_form_submits( $this->entryCount );
	}

	/**
	 * @param $popupFormSubmit
	 *
	 * @return int
	 */
	private function processSinglePopupFormSubmit( $popupFormSubmit ) {
		try {
			if ( isset( $popupFormSubmit->checkoutSessionId ) ) {
				$checkoutSession = $this->retrieveCheckoutSession( $popupFormSubmit->checkoutSessionId );
				$paymentIntent   = $this->findPaymentIntentInCheckoutSession( $checkoutSession );
				if ( isset( $paymentIntent ) && \Stripe\PaymentIntent::STATUS_SUCCEEDED === $paymentIntent->status ) {
					$formModel             = null;
					$checkoutChargeHandler = null;
					if (
						MM_WPFS::FORM_TYPE_POPUP_PAYMENT === $popupFormSubmit->formType
						|| MM_WPFS::FORM_TYPE_POPUP_SAVE_CARD === $popupFormSubmit->formType
					) {
						$formModel             = new MM_WPFS_Public_PopupPaymentFormModel();
						$checkoutChargeHandler = new MM_WPFS_CheckoutPaymentChargeHandler();
					} elseif ( MM_WPFS::FORM_TYPE_POPUP_SUBSCRIPTION === $popupFormSubmit->formType ) {
						$formModel             = new MM_WPFS_Public_PopupSubscriptionFormModel();
						$checkoutChargeHandler = new MM_WPFS_CheckoutSubscriptionChargeHandler();
					}
					if ( ! is_null( $formModel ) && ! is_null( $checkoutChargeHandler ) ) {
						$postData            = $formModel->extractFormModelDataFromPopupFormSubmit( $popupFormSubmit );
						$checkoutSessionData = $formModel->extractFormModelDataFromCheckoutSession( $checkoutSession );
						$postData            = array_merge( $postData, $checkoutSessionData );
						$formModel->bindByArray(
							$postData
						);
						if ( $this->debugLog ) {
							MM_WPFS_Utils::log( 'processSinglePopupFormSubmit(): formModel=' . print_r( $formModel, true ) );
						}
						$chargeResult = $checkoutChargeHandler->handle( $formModel, $checkoutSession );
						if ( $chargeResult->isSuccess() ) {
							$this->updateSubmitEntryWithSuccess( $popupFormSubmit, $chargeResult->getMessageTitle(), $chargeResult->getMessage() );

							return self::PROCESS_RESULT_SET_TO_SUCCESS;
						} else {
							$this->updateSubmitEntryWithFailed( $popupFormSubmit );

							return self::PROCESS_RESULT_SET_TO_FAILED;
						}
					} else {
						MM_WPFS_Utils::log( 'processSinglePopupFormSubmit(): Unknown formType=' . $popupFormSubmit->formType );
						$this->updateSubmitEntryWithFailed( $popupFormSubmit );

						return self::PROCESS_RESULT_SET_TO_FAILED;
					}
				}
			}

			// tnagy check expiration
			$expirationDate = time() - 24 * 60 * 60;
			$creationDate   = strtotime( $popupFormSubmit->created );

			if ( $creationDate < $expirationDate ) {
				return self::PROCESS_RESULT_EXPIRED;
			}

		} catch ( Exception $e ) {
			MM_WPFS_Utils::logException( $e, $this );

			return self::PROCESS_RESULT_INTERNAL_ERROR;
		}

		return self::PROCESS_RESULT_WAIT_FOR_STATUS_CHANGE;
	}

	/**
	 * @param $checkoutSessionId
	 *
	 * @return \Stripe\Checkout\Session
	 */
	public function retrieveCheckoutSession( $checkoutSessionId ) {
		$checkoutSession = \Stripe\Checkout\Session::retrieve(
			array(
				'id'     => $checkoutSessionId,
				'expand' => array(
					'customer',
					'payment_intent',
					'payment_intent.payment_method',
					'setup_intent',
					'setup_intent.payment_method',
					'subscription',
					'subscription.latest_invoice.payment_intent',
					'subscription.pending_setup_intent',
				)
			)
		);

		return $checkoutSession;
	}

	/**
	 * @param \Stripe\Checkout\Session $checkoutSession
	 *
	 * @return null|\Stripe\PaymentIntent
	 */
	public function findPaymentIntentInCheckoutSession( $checkoutSession ) {
		$paymentIntent = null;
		if ( isset( $checkoutSession ) ) {
			$paymentIntent = $this->retrieveStripePaymentIntentByCheckoutSession( $checkoutSession );
			if ( is_null( $paymentIntent ) ) {
				$stripeSubscription = $this->retrieveStripeSubscriptionByCheckoutSession( $checkoutSession );
				if ( $this->debugLog ) {
					MM_WPFS_Utils::log( 'findPaymentIntentInCheckoutSession(): subscription=' . print_r( $stripeSubscription, true ) );
				}
				$paymentIntent = $this->findPaymentIntentInSubscription( $stripeSubscription );
				if ( $this->debugLog ) {
					MM_WPFS_Utils::log( 'findPaymentIntentInCheckoutSession(): paymentIntent=' . print_r( $paymentIntent, true ) );
				}
			}
		}

		return $paymentIntent;
	}

	/**
	 * @param \Stripe\Checkout\Session $checkoutSession
	 *
	 * @return \Stripe\PaymentIntent
	 */
	public function retrieveStripePaymentIntentByCheckoutSession( $checkoutSession ) {
		$stripePaymentIntent = null;
		if ( isset( $checkoutSession ) ) {
			if ( isset( $checkoutSession->payment_intent ) ) {
				if ( $checkoutSession->payment_intent instanceof \Stripe\PaymentIntent ) {
					$stripePaymentIntent = $checkoutSession->payment_intent;
				} else {
					$stripePaymentIntent = \Stripe\PaymentIntent::retrieve( $checkoutSession->payment_intent );
				}
			}
		}

		return $stripePaymentIntent;
	}

	/**
	 * @param \Stripe\Checkout\Session $checkoutSession
	 *
	 * @return \Stripe\Subscription
	 */
	public function retrieveStripeSubscriptionByCheckoutSession( $checkoutSession ) {
		$stripeSubscription = null;
		if ( isset( $checkoutSession ) ) {
			if ( isset( $checkoutSession->subscription ) ) {
				if ( $checkoutSession->subscription instanceof \Stripe\Subscription ) {
					$stripeSubscription = $checkoutSession->subscription;
				} else {
					$stripeSubscription = \Stripe\Subscription::retrieve(
						array(
							'id'     => $checkoutSession->subscription,
							'expand' => array(
								'latest_invoice.payment_intent',
								'pending_setup_intent'
							)
						)
					);
				}
			}
		}

		return $stripeSubscription;
	}

	/**
	 * @param \Stripe\Subscription $stripeSubscription
	 *
	 * @return null|\Stripe\PaymentIntent
	 */
	public function findPaymentIntentInSubscription( $stripeSubscription ) {
		$paymentIntent = null;
		if ( isset( $stripeSubscription ) ) {
			if ( isset( $stripeSubscription->latest_invoice ) ) {
				$stripeInvoice = null;
				if ( $stripeSubscription->latest_invoice instanceof \Stripe\Invoice ) {
					$stripeInvoice = $stripeSubscription->latest_invoice;
				} else {
					$stripeInvoice = \Stripe\Invoice::retrieve(
						$stripeSubscription->latest_invoice,
						array(
							'expand' => array(
								'payment_intent'
							)
						)
					);
				}
				if ( isset( $stripeInvoice->payment_intent ) ) {
					if ( $stripeInvoice->payment_intent instanceof \Stripe\PaymentIntent ) {
						$paymentIntent = $stripeInvoice->payment_intent;
					} else {
						$paymentIntent = \Stripe\PaymentIntent::retrieve( $stripeInvoice->payment_intent );
					}
				}
			}
		}

		return $paymentIntent;
	}

	/**
	 * @param $popupFormSubmit
	 * @param string $lastMessageTitle
	 * @param string $lastMessage
	 *
	 * @return false|int
	 */
	public function updateSubmitEntryWithSuccess( $popupFormSubmit, $lastMessageTitle, $lastMessage ) {
		if ( is_null( $popupFormSubmit ) ) {
			return false;
		}

		return $this->db->update_popup_form_submit_by_hash(
			$popupFormSubmit->hash,
			array(
				'status'           => self::POPUP_FORM_SUBMIT_STATUS_SUCCESS,
				'lastMessageTitle' => $lastMessageTitle,
				'lastMessage'      => $lastMessage
			)
		);
	}

	/**
	 * @param $popupFormSubmit
	 * @param $lastMessageTitle
	 * @param $lastMessage
	 *
	 * @return false|int
	 */
	public function updateSubmitEntryWithFailed( $popupFormSubmit, $lastMessageTitle = null, $lastMessage = null ) {
		if ( is_null( $popupFormSubmit ) ) {
			return false;
		}

		if ( is_null( $lastMessageTitle ) ) {
			$lastMessageTitle = __( 'Failed', 'wp-full-stripe' );
		}
		if ( is_null( $lastMessage ) ) {
			$lastMessage = __( 'Payment failed!', 'wp-full-stripe' );
		}

		return $this->db->update_popup_form_submit_by_hash(
			$popupFormSubmit->hash,
			array(
				'status'           => self::POPUP_FORM_SUBMIT_STATUS_FAILED,
				'lastMessageTitle' => $lastMessageTitle,
				'lastMessage'      => $lastMessage
			)
		);
	}

	/**
	 * @param $popupFormSubmit
	 *
	 * @return false|int
	 */
	public function updateSubmitEntryWithCancelled( $popupFormSubmit ) {
		if ( is_null( $popupFormSubmit ) ) {
			return false;
		}

		return $this->db->update_popup_form_submit_by_hash(
			$popupFormSubmit->hash,
			array(
				'status'           => self::CHECKOUT_SESSION_STATUS_CANCELLED,
				'lastMessageTitle' => __( 'Cancelled', 'wp-full-stripe' ),
				'lastMessage'      => __( 'The customer has cancelled the payment.', 'wp-full-stripe' )
			)
		);
	}

	/**
	 * @param $popupFormSubmit
	 *
	 * @return false|int
	 */
	public function updateSubmitEntryWithComplete( $popupFormSubmit ) {
		if ( $this->debugLog ) {
			MM_WPFS_Utils::log( 'updateSubmitEntryWithComplete(): CALLED, popupFormSubmit=' . print_r( $popupFormSubmit, true ) );
		}

		if ( is_null( $popupFormSubmit ) ) {
			return false;
		}

		return $this->db->update_popup_form_submit_by_hash(
			$popupFormSubmit->hash,
			array(
				'status' => self::POPUP_FORM_SUBMIT_STATUS_COMPLETE
			)
		);
	}

	/**
	 * @param $popupFormSubmitIdsToDelete
	 *
	 * @return int
	 */
	private function deleteSubmitEntriesById( $popupFormSubmitIdsToDelete ) {
		$deleted = 0;
		if (
			isset( $popupFormSubmitIdsToDelete )
			&& is_array( $popupFormSubmitIdsToDelete )
			&& sizeof( $popupFormSubmitIdsToDelete ) > 0
		) {
			$deleted = $this->db->delete_popup_form_submits_by_id( $popupFormSubmitIdsToDelete );
		}

		return $deleted;
	}

	private function stop() {
		self::$running = false;
	}

	/**
	 * @param \Stripe\Checkout\Session $checkoutSession
	 *
	 * @return null|\Stripe\SetupIntent
	 */
	public function findSetupIntentInCheckoutSession( $checkoutSession ) {
		$setupIntent = null;
		if ( isset( $checkoutSession ) ) {
			$setupIntent = $this->retrieveStripeSetupIntentByCheckoutSession( $checkoutSession );
			if ( is_null( $setupIntent ) ) {
				$stripeSubscription = $this->retrieveStripeSubscriptionByCheckoutSession( $checkoutSession );
				if ( $this->debugLog ) {
					MM_WPFS_Utils::log( 'findSetupIntentInCheckoutSession(): subscription=' . print_r( $stripeSubscription, true ) );
				}
				$setupIntent = $this->findSetupIntentInSubscription( $stripeSubscription );
				if ( $this->debugLog ) {
					MM_WPFS_Utils::log( 'findSetupIntentInCheckoutSession(): setupIntent=' . print_r( $setupIntent, true ) );
				}
			}
		}

		return $setupIntent;
	}

	/**
	 * @param \Stripe\Checkout\Session $checkoutSession
	 *
	 * @return \Stripe\SetupIntent
	 */
	public function retrieveStripeSetupIntentByCheckoutSession( $checkoutSession ) {
		$stripeSetupIntent = null;
		if ( isset( $checkoutSession ) ) {
			if ( isset( $checkoutSession->setup_intent ) ) {
				if ( $checkoutSession->setup_intent instanceof \Stripe\SetupIntent ) {
					$stripeSetupIntent = $checkoutSession->setup_intent;
				} else {
					$stripeSetupIntent = \Stripe\SetupIntent::retrieve( $checkoutSession->setup_intent );
				}
			}
		}

		return $stripeSetupIntent;
	}

	/**
	 * @param \Stripe\Subscription $stripeSubscription
	 *
	 * @return null|\Stripe\PaymentIntent
	 */
	public function findSetupIntentInSubscription( $stripeSubscription ) {
		$setupIntent = null;
		if ( isset( $stripeSubscription ) ) {
			if ( isset( $stripeSubscription->pending_setup_intent ) ) {
				if ( $stripeSubscription->pending_setup_intent instanceof \Stripe\SetupIntent ) {
					$setupIntent = $stripeSubscription->pending_setup_intent;
				} else {
					$setupIntent = \Stripe\SetupIntent::retrieve( $stripeSubscription->pending_setup_intent );
				}
			}
		}

		return $setupIntent;
	}

	/**
	 * @param \Stripe\Checkout\Session $checkoutSession
	 *
	 * @return \Stripe\Customer
	 */
	public function retrieveStripeCustomerByCheckoutSession( $checkoutSession ) {
		$stripeCustomer = null;
		if ( isset( $checkoutSession ) ) {
			if ( isset( $checkoutSession->customer ) ) {
				if ( $checkoutSession->customer instanceof \Stripe\Customer ) {
					$stripeCustomer = $checkoutSession->customer;
				} else {
					$stripeCustomer = \Stripe\Customer::retrieve( $checkoutSession->customer );
				}
			}
		}

		return $stripeCustomer;
	}

	/**
	 * @param \Stripe\SetupIntent $setupIntent
	 *
	 * @return null|\Stripe\PaymentMethod
	 */
	public function retrieveStripePaymentMethodBySetupIntent( $setupIntent ) {
		$stripePaymentMethod = null;
		if ( isset( $setupIntent ) && $setupIntent instanceof \Stripe\SetupIntent ) {
			if ( isset( $setupIntent->payment_method ) ) {
				if ( $setupIntent->payment_method instanceof \Stripe\PaymentMethod ) {
					$stripePaymentMethod = $setupIntent->payment_method;
				} else {
					$stripePaymentMethod = \Stripe\PaymentMethod::retrieve( $setupIntent->payment_method );
				}
			}
		}

		return $stripePaymentMethod;
	}

	/**
	 * @param \Stripe\PaymentIntent $paymentIntent
	 *
	 * @return null|\Stripe\PaymentMethod
	 */
	public function retrieveStripePaymentMethodByPaymentIntent( $paymentIntent ) {
		$stripePaymentMethod = null;
		if ( isset( $paymentIntent ) && $paymentIntent instanceof \Stripe\PaymentIntent ) {
			if ( isset( $paymentIntent->payment_method ) ) {
				if ( $paymentIntent->payment_method instanceof \Stripe\PaymentMethod ) {
					$stripePaymentMethod = $paymentIntent->payment_method;
				} else {
					$stripePaymentMethod = \Stripe\PaymentMethod::retrieve( $paymentIntent->payment_method );
				}
			}
		}

		return $stripePaymentMethod;
	}

	/**
	 * @param \Stripe\PaymentMethod $paymentMethod
	 *
	 * @return null|\Stripe\Customer
	 */
	public function retrieveStripeCustomerByPaymentMethod( $paymentMethod ) {
		$stripeCustomer = null;
		if ( isset( $paymentMethod ) ) {
			if ( $paymentMethod instanceof \Stripe\PaymentMethod ) {
				if ( isset( $paymentMethod->customer ) ) {
					if ( $paymentMethod->customer instanceof \Stripe\Customer ) {
						$stripeCustomer = $paymentMethod->customer;
					} else {
						$stripeCustomer = \Stripe\Customer::retrieve( $paymentMethod->customer );
					}
				}
			}
		}

		return $stripeCustomer;
	}

}